<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

// Load configuration
require_once 'config.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields
$required = ['from', 'to', 'arrival-date', 'fullname', 'email', 'phone', 'address', 'passengers', 'trip-type'];
$missing = [];
foreach ($required as $field) {
    if (empty($input[$field])) {
        $missing[] = $field;
    }
}

if (!empty($missing)) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields', 'fields' => $missing]);
    exit();
}

// Extract data
$from = htmlspecialchars($input['from']);
$to = htmlspecialchars($input['to']);
$arrivalDate = htmlspecialchars($input['arrival-date']);
$returnDate = !empty($input['return-date']) ? htmlspecialchars($input['return-date']) : null;
$fullname = htmlspecialchars($input['fullname']);
$email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
$phone = htmlspecialchars($input['phone']);
$address = htmlspecialchars($input['address']);
$passengers = htmlspecialchars($input['passengers']);
$tripType = htmlspecialchars($input['trip-type']);

// Validate email
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid email address']);
    exit();
}

try {
    // Format dates
    $arrivalDateFormatted = date('d/m/Y à H:i', strtotime($arrivalDate));
    $returnDateFormatted = $returnDate ? date('d/m/Y à H:i', strtotime($returnDate)) : '';
    $currentDate = date('d/m/Y à H:i');
    
    // Trip type badge
    $tripTypeBadge = $tripType === 'round-trip' 
        ? '<span style="background: linear-gradient(135deg, #4CAF50 0%, #388E3C 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; display: inline-block; box-shadow: 0 3px 8px rgba(76, 175, 80, 0.3);">🔄 Aller/Retour</span>'
        : '<span style="background: linear-gradient(135deg, #FB8C00 0%, #F57C00 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; display: inline-block; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3);">➡️ Aller Simple</span>';
    
    // Generate admin email HTML
    $adminEmailHtml = generateAdminBookingEmail($fullname, $email, $phone, $passengers, $tripTypeBadge, $from, $to, $arrivalDateFormatted, $returnDateFormatted, $address, $currentDate);
    
    // Generate customer email HTML
    $customerEmailHtml = generateCustomerBookingEmail($fullname, $email, $tripTypeBadge, $tripType, $from, $to, $arrivalDateFormatted, $returnDateFormatted, $address, $passengers, $currentDate);
    
    // Send admin email
    sendEmail(
        SMTP_HOST,
        SMTP_PORT,
        EMAIL_USER,
        EMAIL_PASS,
        EMAIL_USER,
        'info@taxiagadirairport.com',
        "🚕 TaxiAgadirAirport - Nouvelle Réservation - {$fullname}",
        $adminEmailHtml,
        $email
    );
    
    // Send customer confirmation email
    sendEmail(
        SMTP_HOST,
        SMTP_PORT,
        EMAIL_USER,
        EMAIL_PASS,
        EMAIL_USER,
        $email,
        '✅ TaxiAgadirAirport - Confirmation de Réservation 🚕',
        $customerEmailHtml,
        'info@taxiagadirairport.com'
    );
    
    // Success response
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Réservation envoyée avec succès!'
    ]);
    
} catch (Exception $e) {
    error_log('Booking error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erreur lors de l\'envoi de la réservation',
        'details' => $e->getMessage()
    ]);
}

// Email generation functions
function generateAdminBookingEmail($fullname, $email, $phone, $passengers, $tripTypeBadge, $from, $to, $arrivalDate, $returnDate, $address, $currentDate) {
    $returnDateRow = $returnDate ? "
    <div style='padding: 12px 0; border-bottom: 1px solid #e9ecef;'>
        <span style='font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;'>Date de retour:</span>
        <span style='color: #1976D2; font-weight: 700; margin-left: 10px;'>{$returnDate}</span>
    </div>
    " : '';
    
    return <<<HTML
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nouvelle Réservation</title>
</head>
<body style="margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6;">
    <div style="width: 100%; background-color: #f0f2f5; padding: 20px 0;">
        <div style="max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15);">
            <div style="background: linear-gradient(135deg, #0D47A1 0%, #1565C0 50%, #1976D2 100%); padding: 40px 30px; text-align: center; position: relative;">
                <div style="position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #FB8C00, #F57C00, #FB8C00);"></div>
                <h1 style="color: #FFFFFF; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.3);">🚕 Taxi Agadir Airport</h1>
                <p style="color: #FFFFFF; margin: 0; font-size: 16px; font-weight: 600;">Nouvelle Demande de Réservation</p>
                <div style="background: #FF5252; color: #FFFFFF; padding: 10px 24px; border-radius: 30px; display: inline-block; font-weight: 700; font-size: 12px; text-transform: uppercase; letter-spacing: 1.2px; margin-top: 15px; box-shadow: 0 4px 10px rgba(255, 82, 82, 0.4);">⚡ URGENT - NOUVELLE RÉSERVATION</div>
                <p style="color: #90CAF9; font-size: 12px; margin-top: 10px; opacity: 0.9;">📅 Reçu le {$currentDate}</p>
            </div>
            
            <div style="padding: 35px 30px;">
                <div style="margin-bottom: 30px;">
                    <div style="background: linear-gradient(135deg, #E3F2FD 0%, #BBDEFB 100%); color: #0D47A1; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #FB8C00;">👤 Informations Client</div>
                    <div style="background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef;">
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Nom complet:</span>
                            <span style="color: #1a1a1a; margin-left: 10px; font-weight: 500;">{$fullname}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Email:</span>
                            <span style="margin-left: 10px;"><a href="mailto:{$email}" style="color: #FB8C00; text-decoration: none; font-weight: 600;">{$email}</a></span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Téléphone:</span>
                            <span style="margin-left: 10px;"><a href="tel:{$phone}" style="color: #FB8C00; text-decoration: none; font-weight: 600;">{$phone}</a></span>
                        </div>
                        <div style="padding: 12px 0;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Passagers:</span>
                            <span style="color: #1976D2; font-weight: 700; margin-left: 10px;">{$passengers}</span>
                        </div>
                    </div>
                </div>
                
                <div style="margin-bottom: 30px;">
                    <div style="background: linear-gradient(135deg, #E3F2FD 0%, #BBDEFB 100%); color: #0D47A1; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #FB8C00;">🚗 Détails du Trajet</div>
                    <div style="background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef;">
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Type:</span>
                            <span style="margin-left: 10px;">{$tripTypeBadge}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Départ:</span>
                            <span style="margin-left: 10px;">{$from}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Destination:</span>
                            <span style="margin-left: 10px;">{$to}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Date d'arrivée:</span>
                            <span style="color: #1976D2; font-weight: 700; margin-left: 10px;">{$arrivalDate}</span>
                        </div>
                        {$returnDateRow}
                        <div style="padding: 12px 0;">
                            <span style="font-weight: 700; color: #0D47A1; background: #E3F2FD; padding: 2px 8px; border-radius: 4px;">Hôtel:</span>
                            <span style="margin-left: 10px;">{$address}</span>
                        </div>
                    </div>
                </div>
                
                <div style="background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 30px;">
                    <p style="margin: 0; color: #BF360C; font-size: 16px; font-weight: 700; margin-bottom: 8px; text-transform: uppercase;">ACTION REQUISE - PRIORITÉ HAUTE</p>
                    <p style="margin: 0; color: #E65100; font-size: 14px; font-weight: 600; line-height: 1.7;">Veuillez contacter le client dans les plus brefs délais pour confirmer la réservation et finaliser les détails du transfert.</p>
                </div>
            </div>
            
            <div style="background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px;">
                <div style="font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00;">🚕 Taxi Agadir Airport</div>
                <div style="height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0;"></div>
                <p style="margin: 8px 0; opacity: 0.9;">📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                <p style="margin: 8px 0; opacity: 0.9;">80010 - Agadir - Maroc</p>
                <p style="margin-top: 12px; opacity: 0.9;">📱 +212 608 003 030 | +212 661 444 189</p>
                <p style="margin: 8px 0; opacity: 0.9;">✉️ <a href="mailto:info@taxiagadirairport.com" style="color: #FB8C00; text-decoration: none; font-weight: 600;">info@taxiagadirairport.com</a></p>
                <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
            </div>
        </div>
    </div>
</body>
</html>
HTML;
}

function generateCustomerBookingEmail($fullname, $email, $tripTypeBadge, $tripType, $from, $to, $arrivalDate, $returnDate, $address, $passengers, $currentDate) {
    $returnDateRow = $returnDate ? "
    <div style='padding: 12px 0; border-bottom: 1px solid #e9ecef;'>
        <span style='font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;'>Date de retour:</span>
        <span style='color: #2E7D32; font-weight: 700; margin-left: 10px;'>{$returnDate}</span>
    </div>
    " : '';
    
    $firstName = explode(' ', $fullname)[0];
    
    return <<<HTML
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Confirmation de Réservation</title>
</head>
<body style="margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6;">
    <div style="width: 100%; background-color: #f0f2f5; padding: 20px 0;">
        <div style="max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15);">
            <div style="background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 50%, #81C784 100%); padding: 40px 30px; text-align: center; position: relative;">
                <div style="position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #FB8C00, #F57C00, #FB8C00);"></div>
                <h1 style="color: white; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2);">🚕 Taxi Agadir Airport</h1>
                <p style="color: #E8F5E9; margin: 0; font-size: 15px; font-weight: 500;">Confirmation de Réservation</p>
                <div style="background: white; color: #4CAF50; padding: 8px 20px; border-radius: 30px; display: inline-block; font-weight: bold; font-size: 11px; text-transform: uppercase; letter-spacing: 1px; margin-top: 15px; box-shadow: 0 4px 10px rgba(255, 255, 255, 0.3);">✅ RÉSERVATION CONFIRMÉE</div>
                <p style="color: #C8E6C9; font-size: 12px; margin-top: 10px; opacity: 0.9;">📅 {$currentDate}</p>
            </div>
            
            <div style="padding: 35px 30px;">
                <div style="background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%); border-left: 5px solid #4CAF50; padding: 20px 24px; border-radius: 12px; margin-bottom: 30px;">
                    <h2 style="color: #2E7D32; font-size: 22px; margin-bottom: 10px;">Merci {$firstName} ! 🎉</h2>
                    <p style="color: #388E3C; font-size: 15px; line-height: 1.7; margin: 0;">Nous avons bien reçu votre demande de réservation. Notre équipe va la traiter dans les plus brefs délais et vous contactera pour confirmer tous les détails de votre transfert.</p>
                </div>

                <div style="margin-bottom: 30px;">
                    <div style="background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%); color: #2E7D32; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #FB8C00;">📋 Récapitulatif de votre réservation</div>
                    <div style="background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef;">
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">Type de trajet:</span>
                            <span style="margin-left: 10px;">{$tripTypeBadge}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">De:</span>
                            <span style="color: #2c3e50; margin-left: 10px; font-weight: 700;">{$from}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">À:</span>
                            <span style="color: #2c3e50; margin-left: 10px; font-weight: 700;">{$to}</span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">Date d'arrivée:</span>
                            <span style="color: #2E7D32; font-weight: 700; margin-left: 10px;">{$arrivalDate}</span>
                        </div>
                        {$returnDateRow}
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">Hôtel:</span>
                            <span style="margin-left: 10px;">{$address}</span>
                        </div>
                        <div style="padding: 12px 0;">
                            <span style="font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px;">Nombre de passagers:</span>
                            <span style="color: #2E7D32; font-weight: 700; margin-left: 10px;">{$passengers}</span>
                        </div>
                    </div>
                </div>

                <div style="background: linear-gradient(135deg, #E3F2FD 0%, #BBDEFB 100%); border: 2px solid #0D47A1; padding: 20px 24px; border-radius: 12px; margin-top: 30px;">
                    <h3 style="color: #0D47A1; font-size: 18px; margin-bottom: 15px;">📌 Prochaines étapes :</h3>
                    <ul style="margin: 0; padding-left: 20px; color: #1565C0;">
                        <li style="margin: 8px 0; line-height: 1.6;"><strong>Confirmation</strong> : Nous vous contacterons sous peu pour confirmer votre réservation</li>
                        <li style="margin: 8px 0; line-height: 1.6;"><strong>Détails du transfert</strong> : Vous recevrez toutes les informations nécessaires (véhicule, chauffeur, prix)</li>
                        <li style="margin: 8px 0; line-height: 1.6;"><strong>Jour du voyage</strong> : Notre chauffeur vous attendra avec une pancarte à votre nom</li>
                        <li style="margin: 8px 0; line-height: 1.6;"><strong>Service 24/7</strong> : Notre équipe est disponible pour toute question</li>
                    </ul>
                </div>

                <div style="background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 25px; text-align: center;">
                    <h3 style="color: #E65100; font-size: 18px; margin-bottom: 12px;">💬 Besoin d'aide ?</h3>
                    <p style="color: #EF6C00; margin: 8px 0; font-size: 15px; font-weight: 600;">Notre équipe est à votre disposition 24/7</p>
                    <p style="color: #EF6C00; margin: 8px 0; font-size: 15px; font-weight: 600;">📱 <a href="tel:+212608003030" style="color: #E65100; text-decoration: none; font-weight: 700;">+212 608 003 030</a> | <a href="tel:+212661444189" style="color: #E65100; text-decoration: none; font-weight: 700;">+212 661 444 189</a></p>
                    <p style="color: #EF6C00; margin: 8px 0; font-size: 15px; font-weight: 600;">✉️ <a href="mailto:info@taxiagadirairport.com" style="color: #E65100; text-decoration: none; font-weight: 700;">info@taxiagadirairport.com</a></p>
                    <p style="margin-top: 12px; font-size: 14px;">
                        <a href="https://wa.me/212661444189" style="background: #25D366; color: white; padding: 10px 20px; border-radius: 25px; text-decoration: none; display: inline-block; margin-top: 10px;">💬 Contactez-nous sur WhatsApp</a>
                    </p>
                </div>
            </div>
            
            <div style="background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px;">
                <div style="font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00;">🚕 Taxi Agadir Airport</div>
                <div style="height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0;"></div>
                <p style="margin: 8px 0; opacity: 0.9;">📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                <p style="margin: 8px 0; opacity: 0.9;">80010 - Agadir - Maroc</p>
                <p style="margin-top: 12px; opacity: 0.9;">📱 +212 608 003 030 | +212 661 444 189</p>
                <p style="margin: 8px 0; opacity: 0.9;">✉️ <a href="mailto:info@taxiagadirairport.com" style="color: #FB8C00; text-decoration: none; font-weight: 600;">info@taxiagadirairport.com</a></p>
                <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
                <p style="margin-top: 15px; font-size: 11px; opacity: 0.6;">Vous recevez cet email car vous avez effectué une demande de réservation sur notre site.</p>
            </div>
        </div>
    </div>
</body>
</html>
HTML;
}

// Email sending function using native PHP mail
function sendEmail($host, $port, $username, $password, $from, $to, $subject, $html, $replyTo) {
    // Set headers
    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "From: Taxi Agadir Airport <{$from}>\r\n";
    $headers .= "Reply-To: {$replyTo}\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    // Send email using native PHP mail
    $success = mail($to, $subject, $html, $headers);
    
    if (!$success) {
        error_log("Email failed to send to: {$to}");
        throw new Exception("Failed to send email");
    }
    
    error_log("Email sent successfully to: {$to}");
}
?>
