const express = require('express');
const cors = require('cors');
const nodemailer = require('nodemailer');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware
app.use(cors({
    origin: ['https://www.taxiagadirairport.com', 'https://taxiagadirairport.com', 'http://localhost:5173', 'http://localhost:3000', 'http://localhost:3001'],
    methods: ['GET', 'POST', 'OPTIONS'],
    credentials: true,
    optionsSuccessStatus: 200
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Create email transporter
const transporter = nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: parseInt(process.env.SMTP_PORT),
    secure: parseInt(process.env.SMTP_PORT) === 465, // true for port 465, false for port 587
    requireTLS: parseInt(process.env.SMTP_PORT) === 587, // Only require TLS for port 587
    auth: {
        user: process.env.EMAIL_USER,
        pass: process.env.EMAIL_PASS
    },
    tls: {
        ciphers: 'SSLv3',
        rejectUnauthorized: false
    },
    debug: true, // Enable debug output
    logger: true // Log to console
});

// Log configuration on startup
console.log('Email Configuration:');
console.log('- Host:', process.env.SMTP_HOST);
console.log('- Port:', process.env.SMTP_PORT);
console.log('- User:', process.env.EMAIL_USER);
console.log('- Secure:', process.env.SMTP_PORT == 587);

// Test SMTP connection on startup
transporter.verify(function(error, success) {
    if (error) {
        console.error('❌ SMTP Connection Error:', error.message);
        console.error('Please check your email configuration!');
    } else {
        console.log('✅ SMTP Server is ready to send emails');
    }
});

// Booking Form Endpoint
app.post('/api/booking', async (req, res) => {
    console.log('=== BOOKING REQUEST RECEIVED ===');
    console.log('Request body:', JSON.stringify(req.body, null, 2));
    console.log('Request headers:', JSON.stringify(req.headers, null, 2));
    
    try {
        const { 
            from, 
            to, 
            'arrival-date': arrivalDate, 
            'return-date': returnDate,
            fullname, 
            email, 
            phone, 
            address, 
            passengers,
            'trip-type': tripType 
        } = req.body;

        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: 'info@taxiagadirairport.com',
            subject: `🚕 TaxiAgadirAirport - Nouvelle Réservation - ${fullname}`,
            html: `
                <!DOCTYPE html>
                <html lang="fr">
                <head>
                    <meta charset="UTF-8">
                    <meta name="viewport" content="width=device-width, initial-scale=1.0">
                    <meta http-equiv="X-UA-Compatible" content="IE=edge">
                    <title>Nouvelle Réservation</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body { margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6; }
                        .email-wrapper { width: 100%; background-color: #f0f2f5; padding: 20px 0; }
                        .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15); }
                        .header { background: linear-gradient(135deg, #0D47A1 0%, #1565C0 50%, #1976D2 100%); padding: 40px 30px; text-align: center; position: relative; }
                        .header::before { content: ''; position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #FB8C00, #F57C00, #FB8C00); }
                        .header h1 { color: #FFFFFF; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.3); }
                        .header p { color: #FFFFFF; margin: 0; font-size: 16px; font-weight: 600; }
                        .header .timestamp { color: #90CAF9; font-size: 12px; margin-top: 10px; opacity: 0.9; }
                        .priority-badge { background: #FF5252; color: #FFFFFF; padding: 10px 24px; border-radius: 30px; display: inline-block; font-weight: 700; font-size: 12px; text-transform: uppercase; letter-spacing: 1.2px; margin-top: 15px; box-shadow: 0 4px 10px rgba(255, 82, 82, 0.4); }
                        .content { padding: 35px 30px; }
                        .section { margin-bottom: 30px; }
                        .section-title { background: linear-gradient(135deg, #E3F2FD 0%, #BBDEFB 100%); color: #0D47A1; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #FB8C00; display: flex; align-items: center; }
                        .section-title::before { content: attr(data-icon); margin-right: 10px; font-size: 22px; }
                        .info-grid { background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef; }
                        .info-row { display: flex; padding: 12px 0; border-bottom: 1px solid #e9ecef; align-items: flex-start; }
                        .info-row:last-child { border-bottom: none; }
                        .info-label { font-weight: 700; color: #0D47A1; min-width: 160px; font-size: 15px; background: #E3F2FD; padding: 2px 8px; border-radius: 4px; }
                        .info-value { color: #1a1a1a; flex: 1; font-size: 15px; word-break: break-word; font-weight: 500; }
                        .info-value a { color: #FB8C00; text-decoration: none; font-weight: 600; }
                        .info-value a:hover { text-decoration: underline; }
                        .info-value strong { color: #1976D2; font-weight: 700; font-size: 15px; }
                        .badge { display: inline-block; background: linear-gradient(135deg, #FB8C00 0%, #F57C00 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; letter-spacing: 0.5px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3); }
                        .badge.round-trip { background: linear-gradient(135deg, #4CAF50 0%, #388E3C 100%); box-shadow: 0 3px 8px rgba(76, 175, 80, 0.3); }
                        .alert-box { background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 30px; box-shadow: 0 4px 12px rgba(251, 140, 0, 0.15); position: relative; }
                        .alert-box::before { content: '⏰'; position: absolute; top: -15px; left: 20px; background: #FB8C00; width: 35px; height: 35px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3); }
                        .alert-box p { margin: 0; color: #E65100; font-size: 14px; font-weight: 600; line-height: 1.7; }
                        .alert-box strong { font-size: 16px; display: block; margin-bottom: 8px; color: #BF360C; text-transform: uppercase; letter-spacing: 0.5px; }
                        .footer { background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px; }
                        .footer-brand { font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00; }
                        .footer p { margin: 8px 0; opacity: 0.9; }
                        .footer a { color: #FB8C00; text-decoration: none; font-weight: 600; }
                        .footer a:hover { text-decoration: underline; }
                        .divider { height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0; }
                        @media only screen and (max-width: 600px) {
                            .container { border-radius: 0; margin: 0; }
                            .content { padding: 25px 20px; }
                            .header { padding: 30px 20px; }
                            .info-row { flex-direction: column; }
                            .info-label { min-width: 100%; margin-bottom: 5px; }
                        }
                    </style>
                </head>
                <body>
                    <div class="email-wrapper">
                    <div class="container">
                        <div class="header">
                            <h1>🚕 Taxi Agadir Airport</h1>
                            <p>Nouvelle Demande de Réservation</p>
                                <div class="priority-badge">⚡ URGENT - NOUVELLE RÉSERVATION</div>
                                <p class="timestamp">📅 Reçu le ${new Date().toLocaleString('fr-FR', { dateStyle: 'full', timeStyle: 'medium' })}</p>
                        </div>
                        
                        <div class="content">
                            <div class="section">
                                    <div class="section-title" data-icon="👤">Informations Client</div>
                                <div class="info-grid">
                                    <div class="info-row">
                                        <span class="info-label">Nom complet:</span>
                                        <span class="info-value">${fullname}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Email:</span>
                                        <span class="info-value"><a href="mailto:${email}" style="color: #FB8C00;">${email}</a></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Téléphone:</span>
                                        <span class="info-value"><a href="tel:${phone}" style="color: #FB8C00;">${phone}</a></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Nombre de personnes:</span>
                                        <span class="info-value"><strong>${passengers}</strong> passager(s)</span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="section">
                                <div class="section-title" data-icon="🚗">Détails du Trajet</div>
                                <div class="info-grid">
                                    <div class="info-row">
                                        <span class="info-label">Type de trajet:</span>
                                        <span class="info-value"><span class="badge ${tripType === 'round-trip' ? 'round-trip' : ''}">${tripType === 'round-trip' ? '🔄 Aller/Retour' : '➡️ Aller Simple'}</span></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Départ:</span>
                                        <span class="info-value">${from}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Destination:</span>
                                        <span class="info-value">${to}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Date d'arrivée:</span>
                                        <span class="info-value"><strong>${new Date(arrivalDate).toLocaleString('fr-FR')}</strong></span>
                                    </div>
                                    ${returnDate ? `
                                    <div class="info-row">
                                        <span class="info-label">Date de retour:</span>
                                        <span class="info-value"><strong>${new Date(returnDate).toLocaleString('fr-FR')}</strong></span>
                                    </div>
                                    ` : ''}
                                    <div class="info-row">
                                        <span class="info-label">Hôtel:</span>
                                        <span class="info-value">${address}</span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="alert-box">
                                <p>
                                    <strong>Action requise - PRIORITÉ HAUTE</strong>
                                    Veuillez contacter le client dans les plus brefs délais pour confirmer la réservation et finaliser les détails du transfert.
                                </p>
                            </div>
                        </div>
                        
                        <div class="footer">
                            <div class="footer-brand">🚕 Taxi Agadir Airport</div>
                            <div class="divider"></div>
                            <p>📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                            <p>80010 - Agadir - Maroc</p>
                            <p style="margin-top: 12px;">📱 +212 608 003 030 | +212 661 444 189</p>
                            <p>✉️ <a href="mailto:info@taxiagadirairport.com">info@taxiagadirairport.com</a></p>
                            <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
                        </div>
                        </div>
                    </div>
                </body>
                </html>
            `,
            replyTo: email
        };

        console.log('Attempting to send booking email to admin...');
        await transporter.sendMail(mailOptions);
        console.log('Booking email sent to admin successfully!');

        // Send confirmation email to customer
        const customerMailOptions = {
            from: process.env.EMAIL_USER,
            to: email,
            subject: `✅ TaxiAgadirAirport - Confirmation de Réservation 🚕`,
            html: `
                <!DOCTYPE html>
                <html lang="fr">
                <head>
                    <meta charset="UTF-8">
                    <meta name="viewport" content="width=device-width, initial-scale=1.0">
                    <meta http-equiv="X-UA-Compatible" content="IE=edge">
                    <title>Confirmation de Réservation</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body { margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6; }
                        .email-wrapper { width: 100%; background-color: #f0f2f5; padding: 20px 0; }
                        .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15); }
                        .header { background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 50%, #81C784 100%); padding: 40px 30px; text-align: center; position: relative; }
                        .header::before { content: ''; position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #FB8C00, #F57C00, #FB8C00); }
                        .header h1 { color: white; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2); }
                        .header p { color: #E8F5E9; margin: 0; font-size: 15px; font-weight: 500; }
                        .header .timestamp { color: #C8E6C9; font-size: 12px; margin-top: 10px; opacity: 0.9; }
                        .success-badge { background: white; color: #4CAF50; padding: 8px 20px; border-radius: 30px; display: inline-block; font-weight: bold; font-size: 11px; text-transform: uppercase; letter-spacing: 1px; margin-top: 15px; box-shadow: 0 4px 10px rgba(255, 255, 255, 0.3); }
                        .content { padding: 35px 30px; }
                        .section { margin-bottom: 30px; }
                        .welcome-message { background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%); border-left: 5px solid #4CAF50; padding: 20px 24px; border-radius: 12px; margin-bottom: 30px; }
                        .welcome-message h2 { color: #2E7D32; font-size: 22px; margin-bottom: 10px; }
                        .welcome-message p { color: #388E3C; font-size: 15px; line-height: 1.7; margin: 0; }
                        .section-title { background: linear-gradient(135deg, #E8F5E9 0%, #C8E6C9 100%); color: #2E7D32; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #FB8C00; display: flex; align-items: center; }
                        .section-title::before { content: attr(data-icon); margin-right: 10px; font-size: 22px; }
                        .info-grid { background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef; }
                        .info-row { display: flex; padding: 12px 0; border-bottom: 1px solid #e9ecef; align-items: flex-start; }
                        .info-row:last-child { border-bottom: none; }
                        .info-label { font-weight: 600; color: #4CAF50; min-width: 160px; font-size: 14px; }
                        .info-value { color: #2c3e50; flex: 1; font-size: 14px; word-break: break-word; }
                        .info-value strong { color: #2E7D32; font-weight: 700; font-size: 15px; }
                        .badge { display: inline-block; background: linear-gradient(135deg, #FB8C00 0%, #F57C00 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; letter-spacing: 0.5px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3); }
                        .badge.round-trip { background: linear-gradient(135deg, #4CAF50 0%, #388E3C 100%); box-shadow: 0 3px 8px rgba(76, 175, 80, 0.3); }
                        .next-steps { background: linear-gradient(135deg, #E3F2FD 0%, #BBDEFB 100%); border: 2px solid #0D47A1; padding: 20px 24px; border-radius: 12px; margin-top: 30px; }
                        .next-steps h3 { color: #0D47A1; font-size: 18px; margin-bottom: 15px; }
                        .next-steps ul { margin: 0; padding-left: 20px; color: #1565C0; }
                        .next-steps li { margin: 8px 0; line-height: 1.6; }
                        .contact-box { background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 25px; text-align: center; }
                        .contact-box h3 { color: #E65100; font-size: 18px; margin-bottom: 12px; }
                        .contact-box p { color: #EF6C00; margin: 8px 0; font-size: 15px; font-weight: 600; }
                        .contact-box a { color: #E65100; text-decoration: none; font-weight: 700; }
                        .footer { background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px; }
                        .footer-brand { font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00; }
                        .footer p { margin: 8px 0; opacity: 0.9; }
                        .footer a { color: #FB8C00; text-decoration: none; font-weight: 600; }
                        .footer a:hover { text-decoration: underline; }
                        .divider { height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0; }
                        @media only screen and (max-width: 600px) {
                            .container { border-radius: 0; margin: 0; }
                            .content { padding: 25px 20px; }
                            .header { padding: 30px 20px; }
                            .info-row { flex-direction: column; }
                            .info-label { min-width: 100%; margin-bottom: 5px; }
                        }
                    </style>
                </head>
                <body>
                    <div class="email-wrapper">
                        <div class="container">
                            <div class="header">
                                <h1>🚕 Taxi Agadir Airport</h1>
                                <p>Confirmation de Réservation</p>
                                <div class="success-badge">✅ RÉSERVATION CONFIRMÉE</div>
                                <p class="timestamp">📅 ${new Date().toLocaleString('fr-FR', { dateStyle: 'full', timeStyle: 'medium' })}</p>
                            </div>
                            
                            <div class="content">
                                <div class="welcome-message">
                                    <h2>Merci ${fullname.split(' ')[0]} ! 🎉</h2>
                                    <p>Nous avons bien reçu votre demande de réservation. Notre équipe va la traiter dans les plus brefs délais et vous contactera pour confirmer tous les détails de votre transfert.</p>
                                </div>

                                <div class="section">
                                    <div class="section-title" data-icon="📋">Récapitulatif de votre réservation</div>
                                    <div class="info-grid">
                                        <div class="info-row">
                                            <span class="info-label">Type de trajet:</span>
                                            <span class="info-value"><span class="badge ${tripType === 'round-trip' ? 'round-trip' : ''}">${tripType === 'round-trip' ? '🔄 Aller/Retour' : '➡️ Aller Simple'}</span></span>
                                        </div>
                                        <div class="info-row">
                                            <span class="info-label">De:</span>
                                            <span class="info-value"><strong>${from}</strong></span>
                                        </div>
                                        <div class="info-row">
                                            <span class="info-label">À:</span>
                                            <span class="info-value"><strong>${to}</strong></span>
                                        </div>
                                        <div class="info-row">
                                            <span class="info-label">Date d'arrivée:</span>
                                            <span class="info-value"><strong>${new Date(arrivalDate).toLocaleString('fr-FR')}</strong></span>
                                        </div>
                                        ${returnDate ? `
                                        <div class="info-row">
                                            <span class="info-label">Date de retour:</span>
                                            <span class="info-value"><strong>${new Date(returnDate).toLocaleString('fr-FR')}</strong></span>
                                        </div>
                                        ` : ''}
                                        <div class="info-row">
                                            <span class="info-label">Hôtel:</span>
                                            <span class="info-value">${address}</span>
                                        </div>
                                        <div class="info-row">
                                            <span class="info-label">Nombre de passagers:</span>
                                            <span class="info-value"><strong>${passengers}</strong> personne(s)</span>
                                        </div>
                                    </div>
                                </div>

                                <div class="next-steps">
                                    <h3>📌 Prochaines étapes :</h3>
                                    <ul>
                                        <li><strong>Confirmation</strong> : Nous vous contacterons sous peu pour confirmer votre réservation</li>
                                        <li><strong>Détails du transfert</strong> : Vous recevrez toutes les informations nécessaires (véhicule, chauffeur, prix)</li>
                                        <li><strong>Jour du voyage</strong> : Notre chauffeur vous attendra avec une pancarte à votre nom</li>
                                        <li><strong>Service 24/7</strong> : Notre équipe est disponible pour toute question</li>
                                    </ul>
                                </div>

                                <div class="contact-box">
                                    <h3>💬 Besoin d'aide ?</h3>
                                    <p>Notre équipe est à votre disposition 24/7</p>
                                    <p>📱 <a href="tel:+212608003030">+212 608 003 030</a> | <a href="tel:+212661444189">+212 661 444 189</a></p>
                                    <p>✉️ <a href="mailto:info@taxiagadirairport.com">info@taxiagadirairport.com</a></p>
                                    <p style="margin-top: 12px; font-size: 14px;">
                                        <a href="https://wa.me/212661444189" style="background: #25D366; color: white; padding: 10px 20px; border-radius: 25px; text-decoration: none; display: inline-block; margin-top: 10px;">
                                            💬 Contactez-nous sur WhatsApp
                                        </a>
                                    </p>
                                </div>
                            </div>
                            
                            <div class="footer">
                                <div class="footer-brand">🚕 Taxi Agadir Airport</div>
                                <div class="divider"></div>
                                <p>📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                                <p>80010 - Agadir - Maroc</p>
                                <p style="margin-top: 12px;">📱 +212 608 003 030 | +212 661 444 189</p>
                                <p>✉️ <a href="mailto:info@taxiagadirairport.com">info@taxiagadirairport.com</a></p>
                                <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
                                <p style="margin-top: 15px; font-size: 11px; opacity: 0.6;">Vous recevez cet email car vous avez effectué une demande de réservation sur notre site.</p>
                            </div>
                        </div>
                    </div>
                </body>
                </html>
            `,
            replyTo: 'info@taxiagadirairport.com'
        };

        console.log('Attempting to send confirmation email to customer...');
        await transporter.sendMail(customerMailOptions);
        console.log('Confirmation email sent to customer successfully!');
        
        res.status(200).json({ 
            success: true, 
            message: 'Réservation envoyée avec succès!' 
        });
    } catch (error) {
        console.error('Error sending booking email:', error);
        console.error('Error details:', error.message);
        res.status(500).json({ 
            success: false, 
            message: 'Erreur lors de l\'envoi de la réservation',
            error: error.message 
        });
    }
});

// Contact Form Endpoint
app.post('/api/contact', async (req, res) => {
    console.log('=== CONTACT REQUEST RECEIVED ===');
    console.log('Request body:', JSON.stringify(req.body, null, 2));
    console.log('Request headers:', JSON.stringify(req.headers, null, 2));
    
    try {
        const { name, email, phone, subject, message } = req.body;

        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: 'info@taxiagadirairport.com',
            subject: `💬 TaxiAgadirAirport - Contact: ${subject} - ${name}`,
            html: `
                <!DOCTYPE html>
                <html lang="fr">
                <head>
                    <meta charset="UTF-8">
                    <meta name="viewport" content="width=device-width, initial-scale=1.0">
                    <meta http-equiv="X-UA-Compatible" content="IE=edge">
                    <title>Nouveau Message de Contact</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body { margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6; }
                        .email-wrapper { width: 100%; background-color: #f0f2f5; padding: 20px 0; }
                        .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15); }
                        .header { background: linear-gradient(135deg, #FB8C00 0%, #F57C00 50%, #EF6C00 100%); padding: 40px 30px; text-align: center; position: relative; }
                        .header::before { content: ''; position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #0D47A1, #1976D2, #0D47A1); }
                        .header h1 { color: white; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2); }
                        .header p { color: #FFF3E0; margin: 0; font-size: 15px; font-weight: 500; }
                        .header .timestamp { color: #FFE0B2; font-size: 12px; margin-top: 10px; opacity: 0.9; }
                        .priority-badge { background: #1976D2; color: white; padding: 8px 20px; border-radius: 30px; display: inline-block; font-weight: bold; font-size: 11px; text-transform: uppercase; letter-spacing: 1px; margin-top: 15px; box-shadow: 0 4px 10px rgba(25, 118, 210, 0.3); }
                        .content { padding: 35px 30px; }
                        .section { margin-bottom: 30px; }
                        .section-title { background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); color: #E65100; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #0D47A1; display: flex; align-items: center; }
                        .section-title::before { content: attr(data-icon); margin-right: 10px; font-size: 22px; }
                        .info-grid { background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef; }
                        .info-row { display: flex; padding: 12px 0; border-bottom: 1px solid #e9ecef; align-items: flex-start; }
                        .info-row:last-child { border-bottom: none; }
                        .info-label { font-weight: 600; color: #FB8C00; min-width: 140px; font-size: 14px; }
                        .info-value { color: #2c3e50; flex: 1; font-size: 14px; word-break: break-word; }
                        .info-value a { color: #0D47A1; text-decoration: none; font-weight: 600; }
                        .info-value a:hover { text-decoration: underline; }
                        .badge { display: inline-block; background: linear-gradient(135deg, #0D47A1 0%, #1976D2 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; letter-spacing: 0.5px; box-shadow: 0 3px 8px rgba(13, 71, 161, 0.3); }
                        .message-box { background: white; border: 2px solid #E0E0E0; padding: 24px 26px; border-radius: 12px; margin-top: 15px; box-shadow: 0 2px 8px rgba(0,0,0,0.08); position: relative; }
                        .message-box::before { content: '💬'; position: absolute; top: -15px; left: 20px; background: linear-gradient(135deg, #FB8C00, #F57C00); width: 35px; height: 35px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3); }
                        .message-box p { margin: 0; color: #424242; line-height: 1.9; white-space: pre-wrap; font-size: 15px; padding-top: 5px; }
                        .alert-box { background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 30px; box-shadow: 0 4px 12px rgba(251, 140, 0, 0.15); position: relative; }
                        .alert-box::before { content: '⏰'; position: absolute; top: -15px; left: 20px; background: #FB8C00; width: 35px; height: 35px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3); }
                        .alert-box p { margin: 0; color: #E65100; font-size: 14px; font-weight: 600; line-height: 1.7; }
                        .alert-box strong { font-size: 16px; display: block; margin-bottom: 8px; color: #BF360C; text-transform: uppercase; letter-spacing: 0.5px; }
                        .footer { background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px; }
                        .footer-brand { font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00; }
                        .footer p { margin: 8px 0; opacity: 0.9; }
                        .footer a { color: #FB8C00; text-decoration: none; font-weight: 600; }
                        .footer a:hover { text-decoration: underline; }
                        .divider { height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0; }
                        @media only screen and (max-width: 600px) {
                            .container { border-radius: 0; margin: 0; }
                            .content { padding: 25px 20px; }
                            .header { padding: 30px 20px; }
                            .info-row { flex-direction: column; }
                            .info-label { min-width: 100%; margin-bottom: 5px; }
                        }
                    </style>
                </head>
                <body>
                    <div class="email-wrapper">
                    <div class="container">
                        <div class="header">
                            <h1>💬 Taxi Agadir Airport</h1>
                            <p>Nouveau Message de Contact</p>
                                <div class="priority-badge">📧 NOUVEAU MESSAGE</div>
                                <p class="timestamp">📅 Reçu le ${new Date().toLocaleString('fr-FR', { dateStyle: 'full', timeStyle: 'medium' })}</p>
                        </div>
                        
                        <div class="content">
                            <div class="section">
                                    <div class="section-title" data-icon="👤">Informations du Contact</div>
                                <div class="info-grid">
                                    <div class="info-row">
                                        <span class="info-label">Nom:</span>
                                        <span class="info-value"><strong>${name}</strong></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Email:</span>
                                        <span class="info-value"><a href="mailto:${email}" style="color: #FB8C00;">${email}</a></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Téléphone:</span>
                                        <span class="info-value"><a href="tel:${phone}" style="color: #FB8C00;">${phone}</a></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Sujet:</span>
                                        <span class="info-value"><span class="badge">${subject}</span></span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="section">
                                <div class="section-title" data-icon="📝">Message</div>
                                <div class="message-box">
                                    <p>${message}</p>
                                </div>
                            </div>
                            
                            <div class="alert-box">
                                <p>
                                    <strong>Action requise - RÉPONDRE RAPIDEMENT</strong>
                                    Veuillez répondre au client dans les plus brefs délais pour assurer un service de qualité.
                                </p>
                            </div>
                        </div>
                        
                        <div class="footer">
                            <div class="footer-brand">🚕 Taxi Agadir Airport</div>
                            <div class="divider"></div>
                            <p>📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                            <p>80010 - Agadir - Maroc</p>
                            <p style="margin-top: 12px;">📱 +212 608 003 030 | +212 661 444 189</p>
                            <p>✉️ <a href="mailto:info@taxiagadirairport.com">info@taxiagadirairport.com</a></p>
                            <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
                        </div>
                        </div>
                    </div>
                </body>
                </html>
            `,
            replyTo: email
        };

        console.log('Attempting to send contact email...');
        await transporter.sendMail(mailOptions);
        console.log('Contact email sent successfully!');
        
        res.status(200).json({ 
            success: true, 
            message: 'Message envoyé avec succès!' 
        });
    } catch (error) {
        console.error('Error sending contact email:', error);
        console.error('Error details:', error.message);
        res.status(500).json({ 
            success: false, 
            message: 'Erreur lors de l\'envoi du message',
            error: error.message 
        });
    }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
    res.status(200).json({ 
        status: 'OK', 
        message: 'Server is running',
        smtp_configured: !!(process.env.SMTP_HOST && process.env.EMAIL_USER),
        timestamp: new Date().toISOString()
    });
});

// Error handling middleware
app.use((err, req, res, next) => {
    console.error('Global error handler:', err);
    res.status(500).json({ 
        success: false, 
        message: 'Une erreur serveur est survenue',
        error: process.env.NODE_ENV === 'production' ? 'Internal Server Error' : err.message
    });
});

app.listen(PORT, '0.0.0.0', () => {
    console.log(`Server is running on port ${PORT}`);
    console.log(`Access at: http://localhost:${PORT}`);
    console.log(`Network access: http://192.168.11.134:${PORT}`);
});
